#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/lib/oct/config/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /var/lib/oct/common.sh

########
echo "Setting variables"
########
export APP_SHORT="wordpress"
cockpit_port="9080"
php_version="8.2"

metadata_variable_list=(
	"pmadbpass"
	"pmadbuser"
	"pmamodalpass"
	"wpadminemail"
	"wpadminpass"
	"wpadminuser"
	"wptitle"
)

wp_plugin_list=(
	"wordfence"
	#"smtp-mailer"
)

########
echo "Retrieve necessary metadata values"
######

get_var ip

for var in "${metadata_variable_list[@]}"; do
	get_var "${var}"
done

# Exit if any required variables are not set
: "${ip:?}" \
	"${pmadbpass:?}" "${pmadbuser:?}" "${pmamodalpass:?}" \
	"${wpadminemail:?}" "${wpadminpass:?}" "${wpadminuser:?}" "${wptitle:?}"

########
echo "Download WordPress CLI utility 'wp-cli'"
########
curl -L --silent --output /usr/local/bin/wp-cli "https://github.com/wp-cli/wp-cli/releases/download/v2.7.1/wp-cli-2.7.1.phar"
chown www-data:www-data /usr/local/bin/wp-cli
chmod +x /usr/local/bin/wp-cli

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Install and configure SQL Database"
########
install_mysql "${mysql_ibps:-"0"}" "${mysql_toc:-"0"}"
systemctl enable --now mariadb.service

########
echo "Install and configure PHP"
########
install_php "${php_version}" "optimize"

########
echo "Install and configure nginx"
########
install_nginx

########
echo "Install apache2-utils for htpasswd"
########
install_packages "apache2-utils"

########
echo "Install 'certbot'"
########
install_certbot

########
echo "Install, configure, and secure PHPMyAdmin"
########
install_phpmyadmin "pmauser${pmadbuser}" "${pmadbpass}"
htpasswd -B -i -c /etc/nginx/htpasswd/phpmyadmin "pmauser${pmadbuser}" <<< "${pmamodalpass}" > /dev/null 2>&1

########
echo "Generate TLS certificates"
########
generate_self_signed_cert

########
echo "Install and configure Cockpit"
########
install_cockpit "${ip}" "${cockpit_port}"

########
echo "Install and configure 'maldet'"
########
install_maldet
systemctl enable maldet.service

########
echo "Install and configure WordPress"
########
cp /etc/nginx/orig/wordpress_http{,s}.conf /etc/nginx/sites-available/
ln -s /etc/nginx/sites-{available,enabled}/wordpress_http.conf
ln -s /etc/nginx/sites-{available,enabled}/wordpress_https.conf

(
	cd /var/www/html
	curl --silent --output latest.zip "https://wordpress.org/latest.zip"

	unzip -qq latest.zip
	rm -f latest.zip

	mv wordpress/* .
	rmdir wordpress
)
mkdir -p /root/wp-sensitive-files /var/www/html/wp-content/uploads
mv /var/www/html/license.txt /var/www/html/readme.html /root/wp-sensitive-files/

rm -rf \
	/var/www/html/wp-config-sample.php \
	/var/www/html/wp-content/plugins/akismet \
	/var/www/html/wp-content/plugins/hello.php

chmod 755 /var/www/html/wp-content/uploads
chown -R www-data:www-data /var/www

########
echo "Configure WordPress database"
########
wp_db_name="wp$(generate_random_characters 7 digit)"
wp_db_user="wpuser$(generate_random_characters 5 digit)"
wp_db_pass="$(generate_db_password)"
mysql --defaults-file=/root/.my.cnf -u root -e "CREATE DATABASE ${wp_db_name};"
mysql --defaults-file=/root/.my.cnf -u root -e "CREATE USER '${wp_db_user}'@'localhost' IDENTIFIED BY '${wp_db_pass}';"
mysql --defaults-file=/root/.my.cnf -u root \
	-e "GRANT ALL PRIVILEGES ON ${wp_db_name}.* TO '${wp_db_user}'@'localhost'; FLUSH PRIVILEGES;"

########
echo "Create WordPress configuration"
########
cd /var/www/html || {
    echo "Error: Cannot access /var/www/html"
    exit 1
}

/usr/local/bin/wp-cli config create \
    --allow-root \
	--dbname="${wp_db_name}" \
	--dbuser="${wp_db_user}" \
	--dbpass="${wp_db_pass}" \
	--extra-php << EXTRAPHP
define( 'FORCE_SSL_ADMIN', false );
define( 'FS_METHOD' , 'direct' );
if (! defined('WFWAF_STORAGE_ENGINE')) { define('WFWAF_STORAGE_ENGINE', 'mysqli'); }
EXTRAPHP

########
echo "Complete installation of WordPress via 'wp-cli'"
########
/usr/local/bin/wp-cli core install \
    --allow-root \
	--url="${ip}" \
	--title="${wptitle}" \
	--admin_user="wpauser${wpadminuser}" \
	--admin_password="${wpadminpass}" \
	--admin_email="${wpadminemail}" \
	--skip-email

########
echo "Wait for completion of a successful WordPress installation"
########
#until /usr/local/bin/wp-cli &> /dev/null; do
#	sleep 1
#done

########
echo "Install WordPress plugins"
########
/usr/local/bin/wp-cli plugin install "${wp_plugin_list[@]}" --allow-root --activate

for plugin in "${wp_plugin_list[@]}"; do
	[[ -d "/var/www/html/wp-content/plugins/${plugin}" ]] ||
		{
			printf 'Plugin (%s) not installed!\n' "${plugin}"
			return 255
		}
	printf 'Plugin (%s) installed and activated!\n' "${plugin}"
done

########
echo "Check for WordPress updates"
########
/usr/local/bin/wp-cli --allow-root  core update

########
echo "Ensure correct permissions on web content files"
########
chown -R www-data:www-data /var/www/html/

########
echo "Remove any remaining unnecessary packages"
########
autoremove_packages

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"
cleanup_imageless

########
echo "Enable services"
########
systemctl enable --now \
	cockpit.socket \
	"php${php_version}-fpm.service" \
	nginx.service

########
echo "Configure firewall"
########
ufw default deny incoming
ufw allow 80/tcp
ufw allow 443/tcp
ufw allow "${cockpit_port}/tcp"
ufw status verbose
