#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /opt/oct-installer/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

arch="${OCTENIUM_COMMON_ARCHITECTURE:-$(uname -m)}"
if [[ "${arch}" != arm64 ]]; then
	arch="amd64"
fi

regex="[[:digit:]]+\.[[:digit:]]+(\.[[:digit:]]+)?"
base="https://github.com"

########
echo "Retrieve necessary values from metadata"
########
get_var "prompass"
: "${prompass:?}"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Create user"
########
groupadd --system prometheus
useradd --no-create-home --system --shell /usr/sbin/nologin --gid prometheus prometheus
mkdir -p /etc/prometheus /var/lib/prometheus

########
echo "Install Prometheus"
########
prom_repo="prometheus/prometheus"
prom_assets="$(
	curl -fsSL "${base}/${prom_repo}/releases/latest" |
		grep -oE "${prom_repo}/releases/expanded_assets/v${regex}"
)"

prom_tarball="$(
	curl -fsSL "${base}/${prom_assets}" |
		grep -oE "${prom_repo}/releases/download/v${regex}/prometheus-${regex}.linux-${arch}.tar.gz"
)"

curl -fsSL --output latest-prometheus.tar.gz "${base}/${prom_tarball}"

tar -xzf latest-prometheus.tar.gz
mv ./prometheus-*."linux-${arch}" ./prometheus
mv ./prometheus/prom{etheus,tool} /usr/local/bin/
mv ./prometheus/prometheus.yml /etc/prometheus/

rm -rf ./prometheus latest-prometheus.tar.gz

install_packages apache2-utils

passwd_hash="$(htpasswd -nbBiC 10 prometheus <<< "${prompass}")"
cat > /etc/prometheus/webconfig.yml <<- EOF
	---
	basic_auth_users:
	  prometheus: ${passwd_hash#*:}
EOF

chown -R prometheus:prometheus /etc/prometheus /var/lib/prometheus
mv /opt/oct-installer/prometheus.service /usr/lib/systemd/system/
mv /opt/oct-installer/prometheus_node.service /usr/lib/systemd/system/
########
echo "Install Prometheus node_exporter"
########
ne_repo="prometheus/node_exporter"
ne_assets="$(
	curl -fsSL "${base}/${ne_repo}/releases/latest" |
		grep -oE "${ne_repo}/releases/expanded_assets/v${regex}"
)"

ne_tarball="$(
	curl -fsSL "${base}/${ne_assets}" |
		grep -oE "${ne_repo}/releases/download/v${regex}/node_exporter-${regex}.linux-${arch}.tar.gz"
)"

curl -fsSL --output latest-node_exporter.tar.gz "${base}/${ne_tarball}"

tar -xzf latest-node_exporter.tar.gz
mv ./node_exporter-*."linux-${arch}" ./node_exporter
mv ./node_exporter/node_exporter /usr/local/bin/

rm -rf ./node_exporter latest-node_exporter.tar.gz

sed -i -e '/^\s-\sjob_name: /,$ s/^/# /' /etc/prometheus/prometheus.yml
cat >> /etc/prometheus/prometheus.yml <<- EOF
	  - job_name: 'node_exporter'
	    static_configs:
	      - targets: ['localhost:9100']
EOF

########
echo "Remove any remaining unnecessary packages"
########
autoremove_packages

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"


########
echo "Enable services"
########
initsvc enablenow prometheus_node prometheus

cleanup_imageless
