#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/oct/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Setting variables"
########
export APP_SHORT="prestashop"

cockpit_port="9080"
php_version="8.2"
metadata_variable_list=(
	"boamodaluser"
	"boamodalpass"
	"boemail"
	"bopassword"
	"ip"
	"shop_name"
)

########
echo "Retrieve necessary variables from metadata"
########

for var in "${metadata_variable_list[@]}"; do
	get_var "${var}"
done

# Exit if any required variables are not set
: "${ip:?}" \
	"${boamodaluser:?}" "${boamodalpass:?}" "${boemail:?}" "${bopassword:?}"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Install and configure SQL Database"
########
install_mysql "${mysql_ibps:-"0"}" "${mysql_toc:-"0"}"
systemctl enable --now mariadb.service

########
echo "Install and configure PHP"
########
install_php "${php_version}" "optimize"

########
echo "Install and configure nginx"
########
install_nginx

########
echo "Install 'certbot'"
########
install_certbot

########
echo "Install, configure, and secure PHPMyAdmin"
########
pmadbpass="pma$(generate_random_characters 8 alnum)"
pmadbuser="pma$(generate_random_characters 8 alnum)"
pmamodalpass="$(generate_random_characters 8 alnum)"

install_phpmyadmin "pmauser${pmadbuser}" "${pmadbpass}"
htpasswd -Bic /etc/nginx/htpasswd/phpmyadmin "pmauser${pmadbuser}" <<< "${pmamodalpass}" > /dev/null 2>&1

########
echo "Generate TLS certificates"
########
generate_self_signed_cert

########
echo "Install and configure Cockpit"
########
install_cockpit "${ip}" "${cockpit_port}"

########
echo "Configure PrestaShop Database"
########
ps_db_name="presta$(generate_random_characters 8 alnum)"
ps_db_user="puser$(generate_random_characters 8 alnum)"
ps_db_pass="$(generate_db_password)"
mysql --defaults-file=/root/.my.cnf -u root -e "
	CREATE DATABASE ${ps_db_name} CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci;
	CREATE USER '${ps_db_user}'@'localhost' IDENTIFIED BY '${ps_db_pass}';
	GRANT ALL PRIVILEGES ON ${ps_db_name}.* TO '${ps_db_user}'@'localhost';
	FLUSH PRIVILEGES;
"

########
echo "Install and configure PrestaShop"
########
mkdir -p /etc/nginx/orig/
mv prestashop_http* /etc/nginx/orig/
cp /etc/nginx/orig/prestashop_http{,s}.conf /etc/nginx/sites-available/
ln -s /etc/nginx/sites-{available,enabled}/prestashop_http.conf
ln -s /etc/nginx/sites-{available,enabled}/prestashop_https.conf

(
	cd /var/www/html
	base="https://github.com"
	repo="PrestaShop/PrestaShop"
	regex="[[:digit:]]+\.[[:digit:]]+\.[[:digit:]]+"
	assets="$(curl -fsSL "${base}/${repo}/releases/tag/8.2.1" | grep -oE "${repo}/releases/expanded_assets/${regex}")"
	ps_zip="$(curl -fsSL "${base}/${assets}" | grep -oE "${repo}/releases/download/${regex}/prestashop_${regex}\.zip")"
	curl -fsSL --output latest.zip "${base}/${ps_zip}"

	unzip -qq latest.zip
	rm -f latest.zip index.php Install_PrestaShop.html

	unzip -qq prestashop.zip
	rm -f prestashop.zip

	cp -rp admin admin_area

	ps_db_prefix="ps$(generate_random_characters 8 alnum)_"
	# https://devdocs.prestashop-project.org/8/basics/installation/advanced/install-from-cli/
	php install/index_cli.php \
		--country=us \
		"--db_server=127.0.0.1" \
		"--db_name=${ps_db_name}" \
		"--db_user=${ps_db_user}" \
		"--db_password=${ps_db_pass}" \
		"--domain=${ip}" \
		"--email=${boemail}" \
		--fixtures=0 \
		"--name=${shop_name:-PrestaShop}" \
		"--password=${bopassword}" \
		"--prefix=${ps_db_prefix}" \
		--ssl=1 \
		--timezone=Etc/UTC > /var/log/prestashop-install.log 2>&1

	rm -rf install
)
chown -R www-data:www-data /var/www/html
htpasswd -Bic /etc/nginx/htpasswd/admin_area "ps_bo_user${boamodaluser}" <<< "${boamodalpass}" > /dev/null 2>&1

########
echo "Remove any remaining unnecessary packages"
########
autoremove_packages

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"
cleanup_imageless

########
echo "Enable services"
########
systemctl enable --now \
	cockpit.socket \
	"php${php_version}-fpm.service" \
	nginx.service

########
echo "Configure firewall"
########
ufw default deny incoming
ufw allow 80/tcp
ufw allow 443/tcp
ufw allow "${cockpit_port}/tcp"
ufw status verbose
