#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/oct/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Setting variables"
########
export APP_SHORT="nextcloud"
cockpit_port="9080"

# Per Nextcloud documentation:
# https://docs.nextcloud.com/server/latest/admin_manual/installation/system_requirements.html#server
php_version="8.1"

metadata_variable_list=(
	"adminuser"
	"adminpass"
)

########
echo "Retrieve necessary variables from metadata"
########
get_ip

for var in "${metadata_variable_list[@]}"; do
	get_var "${var}"
done

# Exit if any required variables are not set
: "${ip:?}" \
	"${adminuser:?}" "${adminpass:?}"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Install and configure SQL Database"
########
install_mysql "${mysql_ibps:-"0"}" "${mysql_toc:-"0"}"
systemctl enable --now mariadb.service

########
echo "Install and configure PHP"
########
install_php "${php_version}" "optimize"

########
echo "Install and configure nginx"
########
install_nginx

########
echo "Install 'certbot'"
########
install_certbot

########
echo "Generate TLS certificates"
########
generate_self_signed_cert

########
echo "Install and configure Cockpit"
########
install_cockpit "${ip}" "${cockpit_port}"

########
echo "Install additional dependencies"
########
install_packages \
	"php${php_version}-ldap" \
	"redis-server" \
	"smbclient"

########
echo "Configure redis"
########
systemctl disable --now redis-server.service

mkdir -p /var/default-conf/redis
cp -a /etc/redis/redis.conf /var/default-conf/redis/

sed -i -e 's/^save \(.*\)/#save \1/' /etc/redis/redis.conf
rm -f /var/lib/redis/dump.rdb

########
echo "Download and unpack Nextcloud"
########
mkdir -p /etc/nginx/orig/
mv nextcloud_https.conf /etc/nginx/orig/
cp /etc/nginx/orig/nextcloud_https.conf /etc/nginx/sites-available/
ln -s /etc/nginx/sites-{available,enabled}/nextcloud_https.conf

(
	cd /var/www/html
	curl --fail --silent --show-error --location --output latest.zip \
		https://download.nextcloud.com/server/releases/latest.zip
	curl --fail --silent --show-error --location --output latest.zip.sha256 \
		https://download.nextcloud.com/server/releases/latest.zip.sha256

	sha256sum --check --ignore-missing latest.zip.sha256

	unzip -qq latest.zip
	rm -f latest.zip latest.zip.sha256

	mv nextcloud/* nextcloud/.htaccess .
	rm -rf nextcloud
)
chown -R www-data:www-data /var/www/html

########
echo "Configure additional PHP settings"
########
# Request timeouts and access to paths
sed -i \
	-e 's/.*request_terminate_timeout.*/request_terminate_timeout = 600s/' \
	-e 's/^\;env/env/' \
	"/etc/php/${php_version}/fpm/pool.d/www.conf"

cat >> "/etc/php/${php_version}/fpm/pool.d/www.conf" << EOF
php_admin_value[max_execution_time] = 600
php_admin_value[max_input_time] = 600
php_admin_value[output_buffering] = 0
php_admin_value[memory_limit] = 512M
EOF

# Configure OPCache as recommended by Nextcloud
sed -i \
	-e 's/^;opcache.enable=.*/opcache.enable=1/' \
	-e 's/^;opcache.enable_cli=.*/opcache.enable_cli=1/' \
	-e 's/^;opcache.interned_strings_buffer=.*/opcache.interned_strings_buffer=8/' \
	-e 's/^;opcache.max_accelerated_files=.*/opcache.max_accelerated_files=10000/' \
	-e 's/^;opcache.memory_consumption=.*/opcache.memory_consumption=128/' \
	-e 's/^;opcache.save_comments=.*/opcache.save_comments=1/' \
	-e 's/^;opcache.revalidate_freq=.*/opcache.revalidate_freq=1/' "/etc/php/${php_version}/fpm/php.ini"

########
echo "Configure Nextcloud Database"
########
nc_db_name="nextcloud$(generate_random_characters 5 digit)"
nc_db_user="ncuser$(generate_random_characters 5 digit)"
nc_db_pass="$(generate_db_password)"
mysql --defaults-file=/root/.my.cnf -u root \
	-e "CREATE DATABASE ${nc_db_name} CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci;"
mysql --defaults-file=/root/.my.cnf -u root \
	-e "CREATE USER '${nc_db_user}'@'localhost' IDENTIFIED BY '${nc_db_pass}';"
mysql --defaults-file=/root/.my.cnf -u root \
	-e "GRANT ALL PRIVILEGES ON ${nc_db_name}.* TO '${nc_db_user}'@'localhost'; FLUSH PRIVILEGES;"

########
echo "Complete Nextcloud installation steps"
########
(
	cd /var/www/html
	sudo -u www-data php occ maintenance:install \
		--database "mysql" \
		--database-name "${nc_db_name}" \
		--database-user "${nc_db_user}" \
		--database-pass "${nc_db_pass}" \
		--admin-user "user${adminuser}" \
		--admin-pass "${adminpass}"
)

memcache="'memcache.local' => '\\\OC\\\Memcache\\\Redis',"
memcache="${memcache}\n'redis' => array("
memcache="${memcache}\n'host' => 'localhost',"
memcache="${memcache}\n'port' => 6379,"
memcache="${memcache}\n),"

sed -i \
	-e "/    0 => 'localhost'/a 1 => '${ip}'," \
	-e "/instanceid/a ${memcache}" /var/www/html/config/config.php

########
echo "Enable services"
########
systemctl enable --now \
	cockpit.socket \
	"php${php_version}-fpm.service" \
	nginx.service \
	redis-server.service

########
echo "Cleanup the environment"
########
cleanup_imageless