#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /opt/oct-installer/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Retrieve necessary values from metadata"
########
get_ip
get_var "mysql_password"
: "${ip:?}" "${mysql_password:?}"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Install and configure Apache Web Server, MySQL Database, and PHP runtime"
########
install_packages \
	apache2 \
	libapache2-mod-php \
	mysql-server \
	php-bcmath \
	php-curl \
	php-json \
	php-mbstring \
	php-mysql \
	php-xml \
	php-zip

initsvc disablenow apache2

########
echo "Secure MySQL Database"
########
mysql --defaults-file=/etc/mysql/debian.cnf \
	-e "ALTER USER 'root'@'localhost' IDENTIFIED WITH 'caching_sha2_password' BY '${mysql_password}';"

cat > /root/.my.cnf <<- EOF
	[client]
	user=root
	password=${mysql_password}
EOF

mysql --defaults-file=/root/.my.cnf -e "DELETE FROM mysql.user WHERE User='';"
mysql --defaults-file=/root/.my.cnf -e "DROP DATABASE IF EXISTS test;"
mysql --defaults-file=/root/.my.cnf -e "DELETE FROM mysql.db WHERE Db='test' OR Db='test\\_%';"
mysql --defaults-file=/root/.my.cnf -e "FLUSH PRIVILEGES;"

########
echo "Install Laravel"
########
export HOME=/root COMPOSER_HOME=/root

php -r "copy('https://getcomposer.org/installer', 'composer-setup.php');"
php composer-setup.php --quiet --install-dir=/usr/local/bin --filename=composer
rm -f composer-setup.php

# Create a new Laravel project
COMPOSER_ALLOW_SUPERUSER=1 composer create-project --prefer-dist laravel/laravel /var/www/html/my-laravel-app

# Set permissions for Laravel directories
chown -R www-data:www-data /var/www/html/my-laravel-app
chmod -R 775 /var/www/html/my-laravel-app/bootstrap/cache /var/www/html/my-laravel-app/storage

# Disable the default Apache site
a2dissite 000-default.conf

# Create a virtual host for the Laravel project
cat > /etc/apache2/sites-available/my-laravel-app.conf <<- "EOF"
	<VirtualHost *:80>
	    ServerAdmin webmaster@localhost
	    DocumentRoot /var/www/html/my-laravel-app/public
	    ErrorLog ${APACHE_LOG_DIR}/error.log
	    CustomLog ${APACHE_LOG_DIR}/access.log combined

	    <Directory /var/www/html/my-laravel-app>
	        Options Indexes FollowSymLinks
	        AllowOverride All
	        Require all granted
	    </Directory>

	    <Directory /var/www/html/my-laravel-app/public>
	        Options -Indexes +FollowSymLinks
	        AllowOverride All
	        Require all granted

	        <FilesMatch \.php$>
	            SetHandler application/x-httpd-php
	        </FilesMatch>
	    </Directory>
	</VirtualHost>
EOF

# Enable the new virtual host and rewrite module
a2ensite my-laravel-app
a2enmod rewrite

# Configure MySQL database in .env file
cp /var/www/html/my-laravel-app/{.env.example,.env}
sed -i \
	-e 's/DB_CONNECTION=sqlite/DB_CONNECTION=mysql/' \
	-e 's/SESSION_DRIVER=database/SESSION_DRIVER=file/' /var/www/html/my-laravel-app/.env

# Clear cache
php /var/www/html/my-laravel-app/artisan key:generate
php /var/www/html/my-laravel-app/artisan config:cache

########
echo "Remove any remaining unnecessary packages"
########
autoremove_packages

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"
cleanup_imageless

########
echo "Enable services"
########
initsvc enablenow apache2

########
echo "Cleanup the environment"
########
cleanup_imageless

echo "Installation complete. You can access your Laravel project at http://${ip}"
