#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/lib/oct/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Setting variables"
########
php_version="8.2"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Install and configure SQL Database"
########
install_mysql "${mysql_ibps:-"0"}" "${mysql_toc:-"0"}"
systemctl enable --now mariadb.service

########
echo "Install and configure PHP"
########
install_php "${php_version}" "optimize"

########
echo "Install and configure Apache Web Server"
########
install_packages apache2
systemctl disable --now apache2.service

mkdir -p \
	/var/default-conf/apache2/{html,sites-available} \
	/var/www/html

# Remove any pre-enabled sites
rm -f /etc/apache2/sites-enabled/*

# Backup sites-available and /var/www/html
mv /etc/apache2/sites-available/* /var/default-conf/apache2/sites-available
mv /var/www/html/* /var/default-conf/apache2/html/

# Move our sample configs in place
mv /opt/oct-installer/http{,s}.conf /etc/apache2/sites-available/
mv /opt/oct-installer/index.php /var/www/html/

# Enable apache modules, configs, and sites
a2enmod proxy_fcgi
a2enmod setenvif
a2enmod ssl

a2enconf "php${php_version}-fpm"

a2ensite http
a2ensite https

########
echo "Generate TLS certificates"
########
generate_certs "server" "/CN=localhost" "/etc/apache2/ssl"

########
echo "Remove any remaining unnecessary packages"
########
autoremove_packages

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"

########
echo "Enable services"
########
systemctl enable --now "php${php_version}-fpm.service" apache2.service

########
echo "Cleanup the environment"
########
cleanup_imageless