#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/lib/oct/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Retrieve necessary variables from metadata"
########
get_ip

# Exit if any required variables are not set
: "${ip:?}"

########
echo "Install and configure GitLab"

apt update
apt install -y curl openssh-server ca-certificates tzdata perl openssl

# Add GitLab repo
mkdir -p /etc/apt/keyrings
curl -fsSL https://packages.gitlab.com/gitlab/gitlab-ce/gpgkey | \
gpg --dearmor -o /etc/apt/keyrings/gitlab_gitlab-ce-archive-keyring.gpg

echo "deb [signed-by=/etc/apt/keyrings/gitlab_gitlab-ce-archive-keyring.gpg] https://packages.gitlab.com/gitlab/gitlab-ce/ubuntu noble main" \
> /etc/apt/sources.list.d/gitlab_gitlab-ce.list

apt update

# Create SSL directory
mkdir -p /etc/gitlab/ssl
chmod 755 /etc/gitlab/ssl

# Generate certificate that matches IP
openssl req -x509 -nodes -days 365 -newkey rsa:2048 \
-keyout /etc/gitlab/ssl/${ip}.key \
-out /etc/gitlab/ssl/${ip}.crt \
-subj "/CN=${ip}"

chmod 600 /etc/gitlab/ssl/${ip}.key
chmod 644 /etc/gitlab/ssl/${ip}.crt

# Create a policy-rc.d script to deny start of services
# This prevents the package from starting services before we are ready, avoiding the hang
echo "exit 101" > /usr/sbin/policy-rc.d
chmod +x /usr/sbin/policy-rc.d

# Install GitLab without starting/configuring it immediately
# The policy-rc.d script will block the automatic service start
apt install -y gitlab-ce

# Remove the policy-rc.d script to allow services to start normally later
rm -f /usr/sbin/policy-rc.d

# Configure External URL explicitly
sed -i "s#^external_url .*#external_url 'https://${ip}'#" /etc/gitlab/gitlab.rb

# Reconfigure GitLab to apply changes and start services
gitlab-ctl reconfigure

# Display info
echo ""
echo "=========================================="
echo "GitLab Installation Complete!"
echo "=========================================="
echo "URL: ${ip}"
echo ""
echo "Initial root password:"
grep "Password:" /etc/gitlab/initial_root_password 2>/dev/null | awk '{print $2}' || echo "Check /etc/gitlab/initial_root_password"
echo ""
echo "Common commands:"
echo "  gitlab-ctl status"
echo "  gitlab-ctl restart"
echo "  gitlab-ctl reconfigure"
echo "  gitlab-ctl tail"
echo "=========================================="
########